// foo_input_ofr.cpp contains an OptimFROG Lossless/DualStream audio
// input plug-in for foobar2000 0.9.x playback, with cue sheet support

// Copyright (C) 2005-2006 Florin Ghido
// Version: 1.21b, Date: 2006.06.26


#include "../SDK/foobar2000.h"
#include "../helpers/helpers.h"
#include "../OptimFROG/OptimFROG.h"


using namespace pfc;


// read interface functions begin

class foobar2000_reader_t
{
public:
    service_ptr_t<file> p_file;
    abort_callback* p_abort;
};

#define GET_FILE(instance) (((foobar2000_reader_t*) instance)->p_file)
#define GET_ABORT(instance) (*((foobar2000_reader_t*) instance)->p_abort)

condition_t f_foobar2000_close(void* instance)
{
    return C_TRUE;
}

sInt32_t f_foobar2000_read(void* instance, void* destBuffer, uInt32_t count)
{
    return GET_FILE(instance)->read(destBuffer, count, GET_ABORT(instance));
}

condition_t f_foobar2000_eof(void* instance)
{
    return C_FALSE; // not used internally
}

condition_t f_foobar2000_seekable(void* instance)
{
    return GET_FILE(instance)->can_seek();
}

sInt64_t f_foobar2000_length(void* instance)
{
    return GET_FILE(instance)->get_size(GET_ABORT(instance)); // -1 when unknown
}

sInt64_t f_foobar2000_getPos(void* instance)
{
    return GET_FILE(instance)->get_position(GET_ABORT(instance)); // -1 when unknown
}

condition_t f_foobar2000_seek(void* instance, sInt64_t pos)
{
    GET_FILE(instance)->seek(pos, GET_ABORT(instance));
    return C_TRUE;
}

static ReadInterface rInt_foobar2000 =
{
    f_foobar2000_close,
    f_foobar2000_read,
    f_foobar2000_eof,
    f_foobar2000_seekable,
    f_foobar2000_length,
    f_foobar2000_getPos,
    f_foobar2000_seek
};

// read interface functions end


class input_ofr
{
private:
    pfc::array_t<t_uint8> buffer;
    foobar2000_reader_t reader;
    char* path;

    void* decoderInstance;
    OptimFROG_Info iInfo;

    enum { DELTA_POINTS = 1024 };

public:
    input_ofr()
    {
        reader.p_abort = C_NULL;
        path = C_NULL;
        decoderInstance = OptimFROG_createInstance();
    }

    ~input_ofr()
    {
        if (decoderInstance != C_NULL)
        {
            OptimFROG_destroyInstance(decoderInstance);
            decoderInstance = C_NULL;
        }
    }

    void open(service_ptr_t<file> p_filehint, const char* p_path, t_input_open_reason p_reason, abort_callback& p_abort)
    {
        reader.p_file = p_filehint;
        reader.p_abort = &p_abort;
        path = (char*) p_path;
        input_open_file_helper(reader.p_file, p_path, p_reason, p_abort);

        if (!OptimFROG_openExt(decoderInstance, &rInt_foobar2000, &reader, C_FALSE))
        {
            OptimFROG_destroyInstance(decoderInstance);
            decoderInstance = C_NULL;
            throw exception_io_data();
        }

        OptimFROG_getInfo(decoderInstance, &iInfo);
    }

    void get_info(file_info& p_info, abort_callback& p_abort)
    {
        // read all tags using framework function
        if (reader.p_file->can_seek())
        {
            t_filesize save_pos = reader.p_file->get_position(p_abort);
            try
            {
                tag_processor::read_id3v2_trailing(reader.p_file, p_info, p_abort);
            }
            catch (exception_tag_not_found&)
            {
            }
            reader.p_file->seek(save_pos, p_abort);
        }

        p_info.set_length(audio_math::samples_to_time(iInfo.noPoints, iInfo.samplerate));
        p_info.info_set_int("samplerate", iInfo.samplerate);
        p_info.info_set_int("channels", iInfo.channels);
        p_info.info_set_int("bitspersample", iInfo.bitspersample);
        p_info.info_set_bitrate(iInfo.bitrate);

        if ((path != C_NULL) && (stricmp_utf8(path + strlen(path) - 3, "ofs") == 0))
        {
            p_info.info_set("encoding", "hybrid");
            p_info.info_set("codec", "DualStream");
        }
        else
        {
            p_info.info_set("encoding", "lossless");
            p_info.info_set("codec", "OptimFROG");
        }

        p_info.info_set("mode", iInfo.method);
        p_info.info_set("speedup", iInfo.speedup);
        p_info.info_set_int("version", iInfo.version);
    }

    t_filestats get_file_stats(abort_callback& p_abort)
    {
        return reader.p_file->get_stats(p_abort);
    }

    void decode_initialize(unsigned p_flags, abort_callback& p_abort)
    {
        if (OptimFROG_getPos(decoderInstance) != 0)
        {
            OptimFROG_close(decoderInstance);
            reader.p_file->reopen(p_abort);

            if (!OptimFROG_openExt(decoderInstance, &rInt_foobar2000, &reader, C_FALSE))
            {
                OptimFROG_destroyInstance(decoderInstance);
                decoderInstance = C_NULL;
                throw exception_io_data();
            }
        }
    }

    bool decode_run(audio_chunk& p_chunk, abort_callback& p_abort)
    {
        buffer.set_size(DELTA_POINTS * iInfo.channels * (iInfo.bitspersample / 8));
        sInt32_t pointsRetrieved = OptimFROG_read(decoderInstance, buffer.get_ptr(), DELTA_POINTS);
        if (pointsRetrieved <= 0)
        {
            return false; // EOF or fatal error
        }

        p_chunk.set_data_fixedpoint(buffer.get_ptr(),
            pointsRetrieved * iInfo.channels * (iInfo.bitspersample / 8),
            iInfo.samplerate, iInfo.channels, iInfo.bitspersample,
            audio_chunk::g_guess_channel_config(iInfo.channels));

        return true;
    }

    void decode_seek(double p_seconds, abort_callback& p_abort)
    {
        reader.p_file->ensure_seekable();

        if (!OptimFROG_seekPoint(decoderInstance, audio_math::time_to_samples(p_seconds, iInfo.samplerate)))
        {
            throw exception_io_data();
        }
    }

    bool decode_can_seek()
    {
        return (OptimFROG_seekable(decoderInstance) == C_TRUE);
    }

    bool decode_get_dynamic_info(file_info& p_out, double& p_timestamp_delta)
    {
        return false;
    }

    bool decode_get_dynamic_info_track(file_info& p_out, double& p_timestamp_delta)
    {
        return false;
    }

    void decode_on_idle(abort_callback& p_abort)
    {
        reader.p_file->on_idle(p_abort);
    }

    void retag(const file_info& p_info, abort_callback& p_abort)
    {
        // write APEv2 tags using framework function
        reader.p_file->ensure_seekable();
        tag_processor::write_apev2(reader.p_file, p_info, p_abort);
    }

    static bool g_is_our_content_type(const char* p_content_type)
    {
        return false;
    }

    static bool g_is_our_path(const char* p_path, const char* p_extension)
    {
        return (stricmp_utf8(p_extension, "ofr") == 0) || (stricmp_utf8(p_extension, "ofs") == 0);
    }
};

static input_cuesheet_factory_t<input_ofr> g_input_ofr_factory;

const char* ofr_get_about_message()
{
    static char buffer[1024];
    uInt32_t version = OptimFROG_getVersion();
    sprintf(buffer,
            "foobar2000 OptimFROG input plug-in, version 1.21b [2006.06.26]\n"
            "OptimFROG Lossless/DualStream audio DLL library, version %u.%03u\n"
            "Copyright (C) 1996-2006 Florin Ghido, all rights reserved.\n"
            "Visit http://www.LosslessAudio.org for updates\n"
            "Free for non-commercial use. E-mail: FlorinGhido@yahoo.com",
             version / 1000, version % 1000);
    return buffer;
}

DECLARE_COMPONENT_VERSION("OptimFROG Lossless/DualStream Decoder", "1.21b", ofr_get_about_message());

DECLARE_FILE_TYPE("OptimFROG files", "*.ofr;*.ofs");
